
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* Copyright: 1987-1990 by Apple Computer, Inc., all rights reserved. */

/* fs_ttf.c */

/* load the essential tables of a TT font into memory - SWAP as needed */
#include "fs_itype.h"

/* more swapping macros */
#if (NEED_TO_SWAP)
#define SWAP_TTC_HEADER(p)    swap_ttc_header(p)
#define SWAP_TTF_HEADER(p)    swap_ttf_header(p)
#define SWAP_HEAD(p)        swap_head(p)
#define SWAP_HHEA(p)        swap_hhea(p)
#define SWAP_VHEA(p)        swap_vhea(p)
#define SWAP_MAXP(p)        swap_maxp(p)

#else
#define SWAP_TTC_HEADER(p)
#define SWAP_TTF_HEADER(p)
#define SWAP_HEAD(p)
#define SWAP_HHEA(p)
#define SWAP_VHEA(p)
#define SWAP_MAXP(p)

#endif


#ifdef EDIT_MODE   /* used by TTFEdit only */
FS_SHORT *controlValues;
FS_ULONG controlValueLength;
#endif


/****************************************************************/
/* read from the ttf (FS_FILE or MEMPTR) as appropriate  */
FS_VOID ttf_read_buf(_DS_ TTF *ttf, FS_ULONG offset, FS_ULONG size, FS_BYTE *p)
{
    /* ? read 0 bytes ... just return */
    if (size == 0)
    {
        return;
    }
    if (ttf->memptr)
    {
#ifdef FS_ACT3
        if ((FS_VOID *)(ttf->decomp))
        {
            FS_BYTE *q;
            FS_ULONG save = size;
            q = (FS_BYTE *)MTX_RA_Get_TTF_FragmentPtr( _PS_ ((MTX_RA_TT_Decomp *)(FS_VOID *)(ttf->decomp)), offset, &size);
            if (q == 0)
            {
                SYS_MEMSET(p, 0, save);
                return;
            }
            SYS_MEMCPY(p, q, size);
            MTX_RA_ReleaseTTF_Fragment(_PS_ ((MTX_RA_TT_Decomp *)(FS_VOID *)(ttf->decomp)), q);
        }
        else
#endif /* FS_ACT3 */
            SYS_MEMCPY(p, ttf->memptr + offset + ttf->data_offset, size);
    }
    else if (ttf->fp)
    {
        FS_ULONG r;
        /* do the file IO */
        FS_seek(_PS_ ttf->fp, offset + ttf->data_offset, SEEK_SET);
        r = FS_read(_PS_ ttf->fp, p, size);
        if (r != size)
        {
            SYS_MEMSET(p + r, 0, size - r);
            return;
        }
    }
}

/****************************************************************/
/* same as ttf_read_buf ... but allocate the buffer first */
FS_VOID *ttf_read(_DS_ TTF *ttf, FS_ULONG offset, FS_ULONG size)
{
    FS_BYTE *p = 0;

    /* ? read zero bytes ... just return NULL */
    if (size == 0)
    {
        return 0;
    }
    if (ttf->memptr)
    {
#ifdef FS_ACT3
        if ((FS_VOID *)(ttf->decomp) )
        {
            FS_BYTE *q;
            FS_ULONG save = size;
            q = MTX_RA_Get_TTF_FragmentPtr( _PS_ ((MTX_RA_TT_Decomp *)(FS_VOID *)(ttf->decomp)), offset, &size);
            if (q == 0)
            {
                return 0;
            }
            if (size == save)    /* q is just a pointer into RAM/ROM ... make a copy */
            {
                p = FSS_malloc(_PS_ size);
                if (p)
                {
                    SYS_MEMCPY(p, q, size);
                }
                MTX_RA_ReleaseTTF_Fragment(_PS_ ((MTX_RA_TT_Decomp *)(FS_VOID *)(ttf->decomp)), q);
            }
            else
            {
                /* we already allocated a pointer to hold the expanded TTF data */
                /* p = FSS_malloc(_PS_ size); */
                p = q;
            }
        }
        else
#endif /* FS_ACT3 */
        {
            p = FSS_malloc(_PS_ size);
            if (p)
            {
                SYS_MEMCPY(p, ttf->memptr + offset + ttf->data_offset, size);
            }
        }
    }
    else if (ttf->fp)
    {
#ifdef FS_MAPPED_FONTS
        /* should never get here, indicates a failure to memory map the font */
        STATE.error = ERR_FILE_UNMAP;
        return NULL;
#else
        /* do the file IO */
        p = FSS_malloc(_PS_ size);
        if (p)
        {
            FS_ULONG r;
            FS_seek(_PS_ ttf->fp, offset + ttf->data_offset, SEEK_SET);
            r = FS_read(_PS_ ttf->fp, p, size);
            if (r != size)
            {
                SYS_MEMSET(p, 0, size);
            }
        }
#endif
    }
    if (STATE.error)
    {
        FSS_free(_PS_ p);
        p = 0;
    }
    return (FS_VOID *)p;
}

/****************************************************************/
/* expects <fp> or <memptr> to be valid  ... the code would be   */
/* clearer if the 'act3' table were present in all fonts in a    */
/* TrueType collection ... not just in the 0-th.  Then  we could */
/* simply use "ttf_get_table_info" to determine if we had an     */
/* act3 font ... instead we need to duplicate this code.         */
static int is_act3(_DS_ TTF *ttf)
{
    TTC_HEADER ttc_header;
    TTF_HEADER ttf_header, *p;
    FS_ULONG tag, offset, size;
    int num, i, result;
    FS_ULONG version;

    SYS_MEMSET(&ttc_header, 0, sizeof(TTC_HEADER)); /* coverity UNINIT_USE */
    SYS_MEMSET(&ttf_header, 0, sizeof(TTF_HEADER)); /* coverity UNINIT_USE */

    /* find the start of the first file of a TTC or a TTF */
    /* note we get the 0-th offset as a part of the typedef */
    ttf_read_buf(_PS_ ttf, 0, sizeof(TTC_HEADER), (FS_BYTE *)&ttc_header);
    if (STATE.error)
        return 0;

    tag = SWAPL(ttc_header.tag);
    if (tag == TAG_ttcf)
        offset = SWAPL(ttc_header.offsets[0]);
    else
        offset = 0;

    /* readin the TTF_HEADER */
    size = sizeof(TTF_HEADER);
    ttf_read_buf(_PS_ ttf, offset, size, (FS_BYTE *)&ttf_header);
    if (STATE.error)
        return 0;

    version = SWAPL(ttf_header.version);
    if (version != 0x00010000 && version != 0x74727565 && version != 0x4F54544F)
    {
        STATE.error = ERR_NOT_A_TTF;
        return 0;
    }
    num = SWAPW(ttf_header.numTables);

    /* read in the TTF_HEADER with all table dir's */
    size += (num - 1) * sizeof(TTF_DIR);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_HEADER";
#endif
    p = (TTF_HEADER *)ttf_read(_PS_ ttf, offset, size);
    if (!p || STATE.error)
        return 0;

    /* look for an 'act3' table */
    tag = TAG_act3;
    for (i = 0, result = 0; result == 0 && i < num; i++)
        result = (tag == SWAPL(p->tables[i].tag));

    /* cleanup and leave */
    FSS_free(_PS_ p);
    return result;
}

/****************************************************************/
#ifdef FS_CCC
static FS_SHORT numBitsNeeded(FS_ULONG v)
{
    FS_SHORT numBits = 0;

    while (v)
    {
        numBits++;
        v >>= 1;
    }
    return numBits;
}
#endif

/****************************************************************/
/* because the OS/2 table contains unaligned longs it can not be
* read "nicely", we need to use the GET_xXXXX macros, which can
* read unaligned stuff -- also they get the byte order correct
*/
TTF_OS2 *get_os2_table(_DS_ TTF *ttf)
{
    FS_BYTE *buf;
    FS_ULONG off, len;
    TTF_OS2 *p;

    if (!get_ttf_table_info(_PS_ ttf, TAG_OS2, &off, &len))
        return 0;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_OS2";
#endif
    p = (TTF_OS2 *) FSS_calloc(_PS_ sizeof(TTF_OS2));
    if (p == 0) return 0;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_OS2 buffer";
#endif
    buf = ttf_read(_PS_ ttf, off, len);
    if (buf == 0 || STATE.error)
    {
        FSS_free(_PS_ p);
        return 0;
    }

    get_os2(p, buf, len);

    FSS_free(_PS_ buf);

    return p;
}


FS_VOID get_os2(TTF_OS2 *p, FS_BYTE *buf, FS_ULONG len)
{
    int i;

    p->version = (FS_USHORT)GET_xWORD(buf);
    if (len == 86)
        p->version = 1;  /* force this for bad fonts */
    else if (len == 78)
        p->version = 0;  /* force this for bad fonts */
    buf += 2;
    p->xAvgCharWidth = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->usWeightClass = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    p->usWidthClass = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    p->fsType = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySubscriptXSize = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySubscriptYSize = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySubscriptXOffset = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySubscriptYOffset = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySuperscriptXSize = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySuperscriptYSize = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySuperscriptXOffset = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->ySuperscriptYOffset = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->yStrikeoutSize = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->yStrikeoutPosition = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->sFamilyClass = (FS_SHORT)GET_xWORD(buf);
    buf += 2;

    for (i = 0; i < 10; i++)
        p->panose[i] = *buf++;

    p->ulUnicodeRange1 = (FS_ULONG)GET_xLONG(buf);
    buf += 4;
    p->ulUnicodeRange2 = (FS_ULONG)GET_xLONG(buf);
    buf += 4;
    p->ulUnicodeRange3 = (FS_ULONG)GET_xLONG(buf);
    buf += 4;
    p->ulUnicodeRange4 = (FS_ULONG)GET_xLONG(buf);
    buf += 4;

    for (i = 0; i < 4; i++)
        p->achVendID[i] = *buf++;

    p->fsSelection = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    p->usFirstCharIndex = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    p->usLastCharIndex = (FS_USHORT)GET_xWORD(buf);
    buf += 2;

    /* FYI ... V1.66 has these 3 as FS_USHORT, they really are FS_SHORT */
    p->sTypoAscender = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->sTypoDescender = (FS_SHORT)GET_xWORD(buf);
    buf += 2;
    p->sTypoLineGap = (FS_SHORT)GET_xWORD(buf);
    buf += 2;

    p->usWinAscent = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    p->usWinDescent = (FS_USHORT)GET_xWORD(buf);
    buf += 2;
    if (p->version > 0)
    {
        p->ulCodePageRange1 = (FS_ULONG)GET_xLONG(buf);
        buf += 4;
        p->ulCodePageRange2 = (FS_ULONG)GET_xLONG(buf);
        buf += 4;
        if (p->version > 1)
        {
            p->sxHeight = (FS_SHORT) GET_xWORD(buf);
            buf += 2;
            p->sCapHeight = (FS_SHORT) GET_xWORD(buf);
            buf += 2;
            p->usDefaultChar = (FS_USHORT) GET_xWORD(buf);
            buf += 2;
            p->usBreakChar = (FS_USHORT) GET_xWORD(buf);
            buf += 2;
            p->usMaxContext = (FS_USHORT) GET_xWORD(buf); /*buf +=2;*/
        }
    }
}

/****************************************************************
 * verify that tables in the table directory do not overlap in terms
 * of their offset and length.
 */
static FS_ULONG verifyTableDir(_DS_ TTF *ttf)
{
    FS_USHORT i, j, n = ttf->ttf_header->numTables;
    FS_ULONG offset = 0;
    TTF_DIR *tables = &ttf->ttf_header->tables[0];

    for (i = 0; i < n; i++)
    {
        offset = tables[i].offset;
        for (j = 0; j < n; j++)
        {
            FS_ULONG off = tables[j].offset;
            FS_ULONG len = tables[j].length;
            if (j != i)
            {
                if ( (offset >= off) &&
                        (offset <  off + len) )
                {
                    STATE.error = ERR_BAD_TABLE_DIR;
                    return 0;
                }
                if ( (offset >= off) &&
                        (offset <  off + len) )
                {
                    STATE.error = ERR_BAD_TABLE_DIR;
                    return 0;
                }
            }
        }
    }

    return 1;
}

/*****************************************************************/
/* check whether the offsets in the cmap table fall on short int */
/* boundaries. Format 14 subtables can cause odd offsets which   */
/* can lead to short int alignment problems on some systems      */
static FS_ULONG cmap_offsets_odd(TTF_CMAP *cmap)
{
    FS_LONG i, n_tabs;
    CMAP_TAB *tabs;
    FS_ULONG offset;

    n_tabs = SWAPW(cmap->number);
    tabs = cmap->tables;
    for (i = 0; i < n_tabs; i++)
    {
        /* byte offset to subtable */
        offset = SWAPL(tabs[i].offset);
        if (offset & 1)
            return 1; /* odd-number offset found */
    }
    return 0; /* cmap has no odd-number offsets */
}

/****************************************************************
 * Because format 14 cmap subtables can result in odd byte offsets
 * the cmap table may need to be realigned to have the subtables
 * begin on short int boundaries. This function replaces the
 * current cmap table with one that is aligned on short ints.
 */
#define MAX_CMAP_SUBTABLES 20
static FS_ULONG create_aligned_cmap_table(_DS_ TTF *ttf)
{
    TTF_CMAP *cmap = ttf->cmap;
    TTF_CMAP *new_cmap;
    CMAP_TAB *tabs;
    FS_ULONG offsets[MAX_CMAP_SUBTABLES + 1] = {0};
    FS_LONG sizes[MAX_CMAP_SUBTABLES] = {0};
    FS_USHORT increments[MAX_CMAP_SUBTABLES] = {0};
    FS_LONG i, j, n_tabs;
    FS_ULONG table_offset, table_size;

    n_tabs = SWAPW(cmap->number);
    if (n_tabs > MAX_CMAP_SUBTABLES)
    {
        STATE.error = ERR_INVALID_CMAP;
        return STATE.error;
    }

    tabs = cmap->tables;

    /* find total size of the cmap table */
    get_ttf_table_info(_PS_ ttf, TAG_cmap, &table_offset, &table_size);

    /* find offsets for each subtable */
    for (i = 0; i < n_tabs; i++)
    {
        offsets[i] = SWAPL(tabs[i].offset);
    }

    /* find sizes for each subtable */
    for (i = 0; i < n_tabs; i++)
    {
        sizes[i] = offsets[i + 1] - offsets[i];
        if (sizes[i] < 0) /* always true for last size since offsets initialized to zero */
            sizes[i] = table_size - offsets[i];
    }

    /* find increments to offsets that will place the start   */
    /* of the corresponding subtable on an even byte boundary */
    for (i = 0; i < n_tabs; i++)
    {
        increments[i] = 0;

        for (j = 0; j < n_tabs; j++)
        {
            if ( (offsets[j] < offsets[i]) && /* subtable j is a prior subtable */
                 ((sizes[j] & 1) == 1 ) )     /* subtable j has odd size */
                increments[i] += 1;          /* need to offset this subtable by 1 */
        }
        if ( ((offsets[i] + increments[i]) & 1) == 1 ) /* if incremented offset is still odd */
            increments[i] += 1;                       /* make it even */
    }

    /* now make a new cmap table with incremented offsets that fall on even byte boundaries */
    /* worst case every subtable moves by one */
    new_cmap = FSS_malloc(_PS_ table_size + n_tabs);
    if (new_cmap == NULL)
    {
        return STATE.error;
    }

    /* copy header with original offsets */
    SYS_MEMCPY(new_cmap, ttf->cmap, 2 * sizeof(FS_USHORT) + n_tabs * sizeof(CMAP_TAB));

    /* copy each subtable to revised offset and revise offsets in header */
    {
        FS_BYTE *ntable = (FS_BYTE *)new_cmap;
        FS_BYTE *otable = (FS_BYTE *)ttf->cmap;

        for (i = 0; i < n_tabs; i++)
        {
            CMAP_TAB *tab;
            FS_ULONG new_offset = offsets[i] + increments[i];

            /* copy subtable to incremented location */
            SYS_MEMCPY(ntable + new_offset, otable + offsets[i], sizes[i]);

            /* revise offset in table header */
            tab = (CMAP_TAB *)(ntable + 4 + i * sizeof(CMAP_TAB));
            tab->offset = SWAPL(new_offset);
        }
    }

    /* free old cmap if we own it and replace it */
    if (ttf->own_cmap)
        FSS_free(_PS_ ttf->cmap);
    ttf->cmap = new_cmap;
    ttf->own_cmap = 1; /* we now own it so it will have to be freed later */

    return SUCCESS;
}
#undef MAX_CMAP_SUBTABLES

/****************************************************************/
/* expects <fp> or <memptr> to be valid, and <index> if TTC  */
/* loads the essential data tables into memory - SWAPed as needed */
/* does nothing that depends on character mapping or scale */
TTF *load_ttf(_DS_ FILECHAR *path, FS_BYTE *memptr, FS_ULONG index,
              FS_ULONG data_offset, FS_ULONG data_length, FS_BOOLEAN validate)
{
    FS_ULONG offset, size, tag, num;
    TTF *ttf;
    TTF_HEADER *p;
    int ccc;
    int ddd;
#ifndef FS_STIK
    int stik = 0;
#endif
#ifdef FS_CCC
    int cccVersion;
#endif

    if (validate)
    {
        /* data length is not used and is not even necessarily valid...can be anything
                if(data_length < 1024)
                {
                    STATE.error = ERR_INVALID_FILE_SIZE;
                    return 0;
                }
        */
    }
    size = sizeof(TTF);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF";
#endif
    ttf = (TTF *)FSS_calloc(_PS_ size);
    if (ttf == 0)
        return 0;
    ttf->data_offset = data_offset;
    ttf->data_length = data_length;

    /* setup for ttf_read ... by setting memptr, fp */

    /* memory based font ? */
    if (memptr)
    {
        ttf->memptr = memptr;
#ifdef FS_ACT3
        ttf->decomp = MTX_RA_TT_Create( _PS_ ttf->memptr + data_offset);
        /* might not have enough HEAP to load a valid ACT3 file, but, ttf->decomp==0 */
        if (STATE.error)
        {
            FSS_free(_PS_ ttf);
            return 0;
        }
#endif /* FS_ACT3 */
    }
    /* file based font ? */
    else if (path)
    {
        /* try as vanilla file */
        ttf->path = path;
        ttf->fp = FS_open(_PS_ ttf->path);
        if (ttf->fp == 0)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    /*** now we have enough information to use ttf_read() ***/

    /* see if they are trying to use compressed fonts improperly */
    if (is_act3(_PS_ ttf))
    {
#ifdef FS_ACT3
        if (!ttf->memptr)
        {
            /* we don't support ACT3 disk fonts at this time */
            STATE.error = ERR_ACT3_DISK;
            unload_ttf(_PS_ ttf);
            return 0;
        }
#else
        /* configuration doesn't support ACT3 fonts */
        STATE.error = ERR_ACT3_UNDEF;
        unload_ttf(_PS_ ttf);
        return 0;
#endif
    }

    if (STATE.error)    /* from is_act3() call which returned 0 */
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }

    /* see if it's a TrueTypeCollection */
    size = sizeof(TTC_HEADER);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTC_HEADER";
#endif
    ttf->ttc_header = (TTC_HEADER *) ttf_read(_PS_ ttf, 0, size);
    if (!ttf->ttc_header || STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }

    tag = SWAPL(ttf->ttc_header->tag);
    if (tag == TAG_ttcf)
    {
        TTC_HEADER *ph = ttf->ttc_header;

        /* it's a TTC ... find the real size of header */
        num = SWAPL(ph->numFonts);
        size += (num - 1) * 4;

        /* allocate a real TTF_HEADER */
        FSS_free(_PS_ ph);
#ifdef FS_MEM_DBG
        STATE.memdbgid = "TTC_HEADER";
#endif
        ph = ttf->ttc_header = (TTC_HEADER *)ttf_read(_PS_ ttf, 0, size);
        if (!ph || STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        SWAP_TTC_HEADER(ph);
        if (index >= ph->numFonts)
        {
            STATE.error = ERR_BAD_TTC_INDEX;
            unload_ttf(_PS_ ttf);
            return 0;
        }
        ttf->ttf_offset = ph->offsets[index];
    }
    else
    {
        /* it's not a TTC ... maybe it's a TTF */
        FSS_free(_PS_ ttf->ttc_header);
        ttf->ttf_offset = 0;
        ttf->ttc_header = 0;
    }

    /* Try and load the TTF at ttf->ttf_offset */
    size = sizeof(TTF_HEADER);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_HEADER";
#endif
    ttf->ttf_header = p = (TTF_HEADER *)ttf_read(_PS_ ttf, ttf->ttf_offset, size);
    if (!p || STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    num = SWAPW(p->numTables);

    if (validate)
    {
        if (num == 0 || num > 50)
        {
            unload_ttf(_PS_ ttf);
            STATE.error = ERR_BAD_TABLE_DIR;
            return 0;
        }
    }
    size += (num - 1) * sizeof(TTF_DIR);
    FSS_free(_PS_ p);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_HEADER";
#endif
    ttf->ttf_header = (TTF_HEADER *)ttf_read(_PS_ ttf, ttf->ttf_offset, size);
    if (!ttf->ttf_header || STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    SWAP_TTF_HEADER(ttf->ttf_header);

    if (validate)
    {
        if (!is_act3(_PS_ ttf))
        {
            if (!verifyTableDir(_PS_ ttf))
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
        }
    }

#ifdef FS_CFFR
    ttf->cff = load_cff(_PS_ ttf);
    if (ttf->cff)
    {
        FS_ULONG length, version;
        FS_USHORT tmp;

        /* no 'loca' */

        /* form an abbreviated 'maxp' table */
        ttf->maxp = FSS_malloc(_PS_ sizeof(TTF_MAXP));
        if (!ttf->maxp)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        SYS_MEMSET(ttf->maxp, 0, sizeof(TTF_MAXP));

        get_ttf_table_info(_PS_ ttf, TAG_maxp, &offset, &length);
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&version);
        version = SWAPL(version);
        if (version != 0x00005000)   /* version 0.5 */
            STATE.error = ERR_CFF_MAXP;
        ttf_read_buf(_PS_ ttf, offset + 4, 2, (FS_BYTE *)&tmp);
        ttf->maxp->numGlyphs = SWAPW(tmp);
        ttf->maxp->maxContours = CFF_CONTOURS;
        ttf->maxp->maxPoints = CFF_POINTS;
        ttf->maxp->maxSizeOfInstructions = 3;    /* for RTGAH */
    }
    else
    {
        if (STATE.error == ERR_TABLE_NOT_FOUND)
            STATE.error = SUCCESS; /* ok, not a CFF font */
        else
        {
            /* is a CFF font, but could not load */
            unload_ttf(_PS_ ttf);
            return 0;
        }

        /* need real 'glyf', 'loca' and 'maxp' tables */

        /* load glyf table info */
        if (!get_ttf_table_info(_PS_ ttf, TAG_glyf, &ttf->glyf_offset, &ttf->glyf_size))
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }

        /* load 'loca'  */
        /* sometimes we need to allocate a buffer and read the table */
        if  (!ttf->memptr || ttf->decomp )
        {
            ttf->loca = (FS_ULONG *)get_ttf_table(_PS_ ttf, TAG_loca);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
        }
        else    /* otherwise we can just compute the address */
        {
            FS_ULONG noffset;

            if (!get_ttf_table_info(_PS_ ttf, TAG_loca, &noffset, 0))
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
            ttf->loca = (FS_ULONG *)(ttf->memptr + noffset + ttf->data_offset);
        }

        /* load 'maxp' */
        ttf->maxp = (TTF_MAXP *)get_ttf_table(_PS_ ttf, TAG_maxp);
        if (STATE.error || !ttf->maxp)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        SWAP_MAXP(ttf->maxp);
    }
#endif /* FS_CFFR */


    /* load 'head' */
    ttf->head = (TTF_HEAD *)get_ttf_table(_PS_ ttf, TAG_head);
    if (STATE.error || (ttf->head == 0))
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    SWAP_HEAD(ttf->head);

    ccc = TTF_FORMAT_CCC(ttf->head->glyphDataFormat) || STIK_FORMAT_CCC( ttf->head->glyphDataFormat );

    ddd = TTF_FORMAT_DDD(ttf->head->glyphDataFormat) || STIK_FORMAT_DDD( ttf->head->glyphDataFormat );

    /* verify the magic number */
    if (ttf->head->magicNumber != TTF_MAGIC)
    {
        STATE.error = ERR_NOT_A_TTF;
        unload_ttf(_PS_ ttf);
        return 0;
    }

#ifndef FS_STIK
    stik = ttf->head->glyphDataFormat == STIK_FORMAT ||
           ttf->head->glyphDataFormat == OFFLINE_STIK_FORMAT ||
           STIK_FORMAT_CCC( ttf->head->glyphDataFormat );
    /* can't use a STIK font if FS_STIK is undefined */
    if (stik)
    {
        STATE.error = ERR_STIK_UNDEF;
        unload_ttf(_PS_ ttf);
        return 0;
    }
#endif

#ifndef FS_CCC
    /* can't use a CCC font if FS_CCC is undefined */
    if (ccc || ddd)
    {
        STATE.error = ERR_CCC_UNDEF;
        unload_ttf(_PS_ ttf);
        return 0;
    }
#endif

    /* load 'hhea' */
    ttf->hhea = (TTF_HHEA *)get_ttf_table(_PS_ ttf, TAG_hhea);
    if (STATE.error || !ttf->hhea)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }

    SWAP_HHEA(ttf->hhea);

#ifndef FS_CFFR
    /* load 'maxp' */
    ttf->maxp = (TTF_MAXP *)get_ttf_table(_PS_ ttf, TAG_maxp);
    if (STATE.error || !ttf->maxp)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    SWAP_MAXP(ttf->maxp);
#endif /* FS_CFFR */

    /* get htmx_offset */
    if (!get_ttf_table_info(_PS_ ttf, TAG_hmtx, &ttf->hmtx_offset, 0))
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
#ifdef FS_SPEED_OVER_MEMORY
    /* load 'hmtx'  */
    /* sometimes we need to allocate a buffer and read the table */
    if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
    {
        ttf->hmtx = (FS_ULONG *)get_ttf_table(_PS_ ttf, TAG_hmtx);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }
    else    /* otherwise we can just compute the address */
    {
        ttf->hmtx = (FS_ULONG *)(ttf->memptr + ttf->hmtx_offset + ttf->data_offset);
    }
#endif /* FS_SPEED_OVER_MEMORY */

#ifndef FS_CFFR
    /* load glyf table info */
    if (!get_ttf_table_info(_PS_ ttf, TAG_glyf, &ttf->glyf_offset, &ttf->glyf_size))
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
#endif /* FS_CFFR */

    /* get vhea offset (optional) */
    get_ttf_table_info(_PS_ ttf, TAG_vmtx, &ttf->vmtx_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;    /* table is optional */

    if (ttf->vmtx_offset)
    {
        /* But ... if 'vmtx' is there, we must have 'vhea' */
        ttf->vhea = (TTF_VHEA *)get_ttf_table(_PS_ ttf, TAG_vhea);
        if (STATE.error || !ttf->vhea)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        SWAP_VHEA(ttf->vhea);
    }

#ifdef FS_SPEED_OVER_MEMORY
    if (ttf->vmtx_offset)
    {
        /* load 'vmtx'  */
        /* sometimes we need to allocate a buffer and read the table */
        if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
        {
            ttf->vmtx = (FS_ULONG *)get_ttf_table(_PS_ ttf, TAG_vmtx);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
        }
        else    /* otherwise we can just compute the address */
        {
            ttf->vmtx = (FS_ULONG *)(ttf->memptr + ttf->vmtx_offset + ttf->data_offset);
        }
    }
#endif /* FS_SPEED_OVER_MEMORY */

#ifdef FS_CCC
    if (ccc | ddd)
    {
        ttf->ccc_info.cccVersion7 = false;
        ttf->ccc_info.cccVersion8 = false;
        cccVersion = ttf->head->glyphDataFormat;

        if ( (cccVersion == 0x0200) || (cccVersion == 0x0400) )
            ttf->ccc_info.cccVersion7 = false;
        else
        {
            /* version8 CCC format: 0x0202, 0x0602 and 0x0402 */
            if ( (cccVersion == 0x0202) || (cccVersion == 0x0402) || cccVersion == 0x0602)
                ttf->ccc_info.cccVersion8 = true;
            else
                ttf->ccc_info.cccVersion7 = true;
        }

        if (ddd || ttf->ccc_info.cccVersion8)
            ttf->ccc_info.numBits_SF = 16;
        else
            ttf->ccc_info.numBits_SF = 10;

        if (!ddd)
        {
            FS_SHORT temp;

            /* minimum LSB from hmtx table */
            ttf_read_buf(_PS_ ttf, ttf->hmtx_offset, sizeof(FS_SHORT), (FS_BYTE *)&temp);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
            ttf->ccc_info.minVal_LSB = SWAPW(temp);

            /* num bits for LSB from hmtx table */
            ttf_read_buf(_PS_ ttf, ttf->hmtx_offset + 2, sizeof(FS_SHORT), (FS_BYTE *)&temp);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
            ttf->ccc_info.numBits_LSB = SWAPW(temp);
            if ((ttf->ccc_info.cccVersion7) || (ttf->ccc_info.cccVersion8) )
            {
                /* minimum AW from hmtx table */
                ttf_read_buf(_PS_ ttf, ttf->hmtx_offset + 4, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                if (STATE.error)
                {
                    unload_ttf(_PS_ ttf);
                    return 0;
                }
                ttf->ccc_info.minVal_AW = SWAPW(temp);

                /* num bits for AW from hmtx table */
                ttf_read_buf(_PS_ ttf, ttf->hmtx_offset + 6, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                if (STATE.error)
                {
                    unload_ttf(_PS_ ttf);
                    return 0;
                }
                ttf->ccc_info.numBits_AW = SWAPW(temp);

                if (ttf->vmtx_offset)
                {
                    /* minimum TSB from hmtx table */
                    ttf_read_buf(_PS_ ttf, ttf->vmtx_offset, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                    if (STATE.error)
                    {
                        unload_ttf(_PS_ ttf);
                        return 0;
                    }
                    ttf->ccc_info.minVal_TSB = SWAPW(temp);
                    /* num bits for tsb from vmtx table */
                    ttf_read_buf(_PS_ ttf, ttf->vmtx_offset + 2, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                    if (STATE.error)
                    {
                        unload_ttf(_PS_ ttf);
                        return 0;
                    }
                    ttf->ccc_info.numBits_TSB = SWAPW(temp);

                    /* minimum AH from vmtx table */
                    ttf_read_buf(_PS_ ttf, ttf->vmtx_offset + 4, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                    if (STATE.error)
                    {
                        unload_ttf(_PS_ ttf);
                        return 0;
                    }
                    ttf->ccc_info.minVal_AH = SWAPW(temp);

                    /* num bits for AH from vmtx table */
                    ttf_read_buf(_PS_ ttf, ttf->vmtx_offset + 6, sizeof(FS_SHORT), (FS_BYTE *)&temp);
                    if (STATE.error)
                    {
                        unload_ttf(_PS_ ttf);
                        return 0;
                    }
                    ttf->ccc_info.numBits_AH = SWAPW(temp);
                }
            }
        }
        else /* ddd */
        {
            FS_BYTE temp;

            /* number of widths (stored as minimum LSB) from hmtx table */
            temp = 0;
            ttf_read_buf(_PS_ ttf, ttf->hmtx_offset, sizeof(FS_BYTE), (FS_BYTE *)&temp);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
            ttf->ccc_info.minVal_LSB = (FS_SHORT)temp;

            /* num bits for widths from hmtx table */
            ttf_read_buf(_PS_ ttf, ttf->hmtx_offset + 1, sizeof(FS_BYTE), (FS_BYTE *)&temp);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
            ttf->ccc_info.numBits_AW = temp;

            if (ttf->vmtx_offset)
            {
                /* minimum TSB from hmtx table */
                ttf_read_buf(_PS_ ttf, ttf->vmtx_offset, sizeof(FS_BYTE), (FS_BYTE *)&temp);
                if (STATE.error)
                {
                    unload_ttf(_PS_ ttf);
                    return 0;
                }
                ttf->ccc_info.minVal_TSB = (FS_SHORT)temp;
                /* num bits for tsb from vmtx table */
                ttf_read_buf(_PS_ ttf, ttf->vmtx_offset + 1, sizeof(FS_BYTE), (FS_BYTE *)&temp);
                if (STATE.error)
                {
                    unload_ttf(_PS_ ttf);
                    return 0;
                }

                ttf->ccc_info.numBits_AH = (FS_SHORT)temp;
            }
        } /* else ddd */

        ttf->ccc_info.numBits_LOCA = numBitsNeeded(1 + ttf->glyf_size);
        ttf->ccc_info.numBits_GLYF = numBitsNeeded(ttf->maxp->maxComponentElements);
    }
#endif /* FS_CCC */


#ifndef FS_CFFR
    /* load 'loca'  */
    /* sometimes we need to allocate a buffer and read the table */
    if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
    {
        ttf->loca = (FS_ULONG *)get_ttf_table(_PS_ ttf, TAG_loca);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }
    else    /* otherwise we can just compute the address */
    {
        FS_ULONG noffset;

        if (!get_ttf_table_info(_PS_ ttf, TAG_loca, &noffset, 0))
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        /* ttf->loca = (FS_ULONG *)offset; */
        ttf->loca = (FS_ULONG *)(ttf->memptr + noffset + ttf->data_offset);
    }
#endif /* FS_CFFR */

    /* load 'cmap' */
    /* sometimes we need to allocate a buffer and read the table */
    if  (!ttf->memptr || (FS_VOID *)(ttf->decomp))
    {
        ttf->cmap = (TTF_CMAP *)get_ttf_table(_PS_ ttf, TAG_cmap);
        if (ttf->cmap == NULL || STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        ttf->own_cmap = 1;
        if (cmap_offsets_odd(ttf->cmap))
        {
            /* create a new ttf->cmap with even offsets */
            STATE.error = create_aligned_cmap_table(_PS_ ttf);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
        }
    }
    else    /* otherwise we can just compute the address */
    {
        FS_ULONG noffset;

        if (!get_ttf_table_info(_PS_ ttf, TAG_cmap, &noffset, 0))
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
        ttf->cmap = (TTF_CMAP *)(ttf->memptr + noffset + ttf->data_offset);
        ttf->own_cmap = 0;
        if (cmap_offsets_odd(ttf->cmap))
        {
            /* create a new ttf->cmap with even offsets */
            STATE.error = create_aligned_cmap_table(_PS_ ttf);
            if (STATE.error)
            {
                unload_ttf(_PS_ ttf);
                return 0;
            }
        }
    }

    /* load "cvt"  */
    get_ttf_table_info(_PS_ ttf, TAG_cvt, &offset, &size);
    if (size == 0 || STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;
    ttf->num_cvt = size / 2;
    if (size)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ttf->cvt";
#endif
        ttf->cvt = (FS_SHORT *) ttf_read(_PS_ ttf, offset, size);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    /* load "prep" */
    get_ttf_table_info(_PS_ ttf, TAG_prep, &offset, &size);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is optional */
    ttf->num_prep = size;
    if (size)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ttf->prep";
#endif
        ttf->prep = ttf_read(_PS_ ttf, offset, size);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    /* load "fpgm" */
    get_ttf_table_info(_PS_ ttf, TAG_fpgm, &offset, &size);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is optional */
    ttf->num_fpgm = size;
    if (size)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ttf->fpgm";
#endif
        ttf->fpgm = ttf_read(_PS_ ttf, offset, size);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    size = sizeof(fnt_funcDef) * ttf->maxp->maxFunctionDefs;
    if (size)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "functionDefs";
#endif
        ttf->functionDefs = (fnt_funcDef *)FSS_malloc(_PS_ size);
        if (!ttf->functionDefs)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    size = sizeof(fnt_instrDef) * ttf->maxp->maxInstructionDefs;
    if (size)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "instructionDefs";
#endif
        ttf->instructionDefs = (fnt_instrDef *)FSS_malloc(_PS_ size);
        if (!ttf->instructionDefs)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

    /* set eblc_offset */
    get_ttf_table_info(_PS_ ttf, TAG_EBLC, &ttf->eblc_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
#ifdef FS_EMBEDDED_BITMAP
    else
    {
        /* load the parts of the EBLC table */
        ttf->ttf_EBLC_partial = get_EBLC_partial(_PS_ ttf);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }
#endif    /* FS_EMBEDDED_BITMAP */

    /* set kern_offset */
    get_ttf_table_info(_PS_ ttf, TAG_kern, &ttf->kern_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    /* set icon_offset */
    get_ttf_table_info(_PS_ ttf, TAG_Icon, &ttf->icon_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
#ifdef FS_EDGE_TECH
    /* load 'ADFH' table into memory */
    get_ttf_table_info(_PS_ ttf, TAG_adfh, &offset, &size);
    if (size == 0 || STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;  /* table is optional */
    if (size)
    {
        ttf->adfh = (FS_BYTE *) get_ttf_table(_PS_ ttf, TAG_adfh);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }
#endif

    /* load 'gasp' table */
    get_ttf_table_info(_PS_ ttf, TAG_gasp, &offset, &size);
    if (size == 0 || STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;  /* table is optional */
    if (size)
    {
        ttf->gasp = (FS_BYTE *) get_ttf_table(_PS_ ttf, TAG_gasp);
        if (STATE.error)
        {
            unload_ttf(_PS_ ttf);
            return 0;
        }
    }

#if defined(FS_ACT3) || defined(FS_CCC)
    /* get the "not really a stick" glyph indices */
    ttf->nstk = (FS_USHORT *)get_ttf_table(_PS_ ttf, TAG_nstk);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
#if NEED_TO_SWAP
    if (ttf->nstk)
    {
        FS_USHORT i, hi, *pn = ttf->nstk;

        hi = SWAPW(pn[0]);
        for (i = 0; i <= hi; i++)
        {
            pn[i] = SWAPW(pn[i]);
        }
    }
#endif /* NEED_TO_SWAP */
#endif /* FS_ACT3 || FS_CCC */

    /* load 'OS/2' */
    ttf->os2 = get_os2_table(_PS_ ttf);

    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    /* load 'name' (abbreviated) */
    ttf->name = (TTF_NAME *)get_name(_PS_ ttf);
    if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    get_ttf_table_info(_PS_ ttf, TAG_VDMX, &ttf->vdmx_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }
    get_ttf_table_info(_PS_ ttf, TAG_hdmx, &ttf->hdmx_offset, 0);
    if (STATE.error == ERR_TABLE_NOT_FOUND)
        STATE.error = SUCCESS;        /* this table is OPTIONAL */
    else if (STATE.error)
    {
        unload_ttf(_PS_ ttf);
        return 0;
    }

    return ttf;
}

/****************************************************************/
/* linear search for a table in the <small> directory */
int get_ttf_table_info(_DS_ TTF *ttf, FS_ULONG tag, FS_ULONG *off, FS_ULONG *len)
{
    FS_USHORT i, n = ttf->ttf_header->numTables;
    TTF_DIR *tables = &ttf->ttf_header->tables[0];

    if (len) *len = 0;
    if (off) *off = 0;

    for (i = 0; i < n; i++)
    {
        if (tables[i].tag == tag)
        {
            if (len) *len = tables[i].length;
            if (off) *off = tables[i].offset;
            return 1;
        }
    }
    STATE.error = ERR_TABLE_NOT_FOUND;
    return 0;
}

/****************************************************************/
FS_VOID *get_ttf_table(_DS_ TTF *ttf, FS_ULONG tag)
{
    FS_ULONG off, len;
    FS_BYTE *p;
#ifdef FS_MEM_DBG
    char tagstr[5];
#endif

    if (!get_ttf_table_info(_PS_ ttf, tag, &off, &len))
        return 0;
#ifdef FS_MEM_DBG
    tagstr[0] = (char)((tag >> 24)       );
    tagstr[1] = (char)((tag >> 16) & 0xFF);
    tagstr[2] = (char)((tag >>  8) & 0xFF);
    tagstr[3] = (char)((tag      ) & 0xFF);
    tagstr[4] = 0;
    STATE.memdbgid = tagstr;
#endif
    p = ttf_read(_PS_ ttf, off, len);
    return (FS_VOID *)p;
}

/****************************************************************/
/* unload an TTF -- possibly partially loaded */
FS_VOID unload_ttf(_DS_ TTF *ttf)
{
#ifdef FS_EMBEDDED_BITMAP
    FS_ULONG i;
#endif    /* FS_EMBEDDED_BITMAP */

    if (ttf)
    {
        if (ttf->fp)
            FS_close(_PS_ ttf->fp);

        /* NOTE: we do NOT free the memptr ... ASSUME it's in ROM */
        /* if you have pre-loaded a font into RAM, free it manually */

        if (ttf->ttc_header) FSS_free(_PS_ (TTC_HEADER *)(ttf->ttc_header));
        FSS_free(_PS_ (TTF_HEADER *)(ttf->ttf_header));
        FSS_free(_PS_ (TTF_HEAD *)(ttf->head));
        FSS_free(_PS_ (TTF_HHEA *)(ttf->hhea));
        if (ttf->vhea) FSS_free(_PS_ (TTF_VHEA *)(ttf->vhea));
        FSS_free(_PS_ (TTF_MAXP *)(ttf->maxp));
        if (ttf->cvt) FSS_free(_PS_ (FS_SHORT *)(ttf->cvt));
        if (ttf->prep) FSS_free(_PS_ (FS_BYTE *)(ttf->prep));
        if (ttf->fpgm) FSS_free(_PS_ (FS_BYTE *)(ttf->fpgm));

#ifdef FS_CFFR
        unload_cff(_PS_ ttf->cff);
#endif /* FS_CFFR */
        FSS_free(_PS_ (TTF_OS2 *)(ttf->os2));
        FSS_free(_PS_ (TTF_NAME *)(ttf->name));
        if (ttf->functionDefs) FSS_free(_PS_ (fnt_funcDef *)(ttf->functionDefs));
        if (ttf->instructionDefs) FSS_free(_PS_ (fnt_instrDef *)(ttf->instructionDefs));

        if (ttf->gasp) FSS_free(_PS_ (FS_BYTE *)(ttf->gasp));

#ifdef FS_EDGE_TECH
        FSS_free(_PS_ (FS_BYTE *)(ttf->adfh));
#endif
        /* was 'loca' read into a buffer ? */
        if (!ttf->memptr)
        {
            FSS_free(_PS_  (FS_ULONG *)(ttf->loca));
            ttf->loca = 0;
        }
        /* is it compressed ? */
#ifdef FS_ACT3
        if ((FS_VOID *)(ttf->decomp))
        {
            FSS_free(_PS_ (FS_ULONG *)(ttf->loca));
            ttf->loca = 0;
            MTX_RA_TT_Destroy(_PS_ (FS_VOID *)(ttf->decomp));
        }
#endif /* FS_ACT3 */

        /* was 'cmap' read into a buffer ? */
        if (ttf->own_cmap)
        {
            FSS_free(_PS_ (TTF_CMAP *)(ttf->cmap));
        }

#ifdef FS_SPEED_OVER_MEMORY
        if (!ttf->memptr || ttf->decomp)
        {
            FSS_free(_PS_ (TTF_CMAP *)(ttf->hmtx));
            if (ttf->vmtx_offset)
                FSS_free(_PS_ (TTF_CMAP *)(ttf->vmtx));
        }
#endif

        /* non_stik table? */
        if (ttf->nstk) FSS_free(_PS_ (FS_USHORT *)(ttf->nstk));

#ifdef FS_EMBEDDED_BITMAP
        if ((TTF_EBLC_PARTIAL *)(ttf->ttf_EBLC_partial))
        {
            for (i = 0; i < ttf->ttf_EBLC_partial->numSizes; i++)
                FSS_free(_PS_ (INDEX_SUB_TABLE *)((BITMAP_SIZE_TABLE_ARRAY *)(ttf->ttf_EBLC_partial->bsta)[i].ist));

            FSS_free(_PS_ (BITMAP_SIZE_TABLE_ARRAY *)(ttf->ttf_EBLC_partial->bsta));
            FSS_free(_PS_ (TTF_EBLC_PARTIAL *)(ttf->ttf_EBLC_partial));
        }
#endif    /* FS_EMBEDDED_BITMAP */
        FSS_free(_PS_ ttf);
    }
}

/****************************************************************/
/* return name table data (abbreviated) */
TTF_NAME *get_abbreviated_name(_DS_ FS_BYTE *p)
{
    FS_LONG jj;
    FS_USHORT nrec, ostr;
    FS_ULONG ii, ulnrec;
    FS_BYTE *ptr;
    TTF_NAME *pname;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_NAME";
#endif
    pname = (TTF_NAME *)FSS_calloc(_PS_ sizeof(TTF_NAME));
    if (pname == (TTF_NAME *)0)
        return (TTF_NAME *)0;

    nrec = GET_xWORD((p + NAME_NUM_REC));     /* nr of records to follow  */
    ostr = GET_xWORD((p + NAME_OFF_TO_STRS)); /* offset to string storage */
    ptr = p + NAME_NAMERECS;                  /* The NameRecords */

    pname->copyright[0] = 0;
    pname->font_family_name[0] = 0;
    pname->font_name[0] = 0;

    /* following is trick to fix a Coverity tainted variable complaint */
    ulnrec = nrec;
    if (ulnrec > 65535)
    {
        STATE.error = ERR_BAD_GLYF_FORMAT;
        return NULL;
    }

    /* extract the font name, family name and copyright data from the "name" table - MSFT format */
    for (ii = 0; ii < ulnrec; ii++, ptr += NAME_SIZE_NAMEREC)
    {
        if ((GET_xWORD((ptr + NAME_TAB_PLATID))) == MSFT_ENC  &&
            (GET_xWORD((ptr + NAME_TAB_LANGID))) == MSFT_US_ENGL )
        {
            FS_BYTE *strPtr = p + ostr + GET_xWORD((ptr + NAME_TAB_STROFF));
            FS_SHORT strLen = GET_xWORD((ptr + NAME_TAB_STRLEN)) >> 1;
            if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_Family)
            {
                /* if font family name length is larger than pre-allocated
                 * buffer length, set "too_long" flag
                 */
                if (strLen > MAX_FONT_FAMILY_NAME_LEN - 1)
                    pname->font_family_name_too_long = true;

                /* strings are in Unicode - 2 bytes per char */

                for (jj = 0, strPtr++; jj < strLen; jj++)
                {
                    if (jj < (MAX_FONT_FAMILY_NAME_LEN - 1))
                        pname->font_family_name[jj] = *strPtr;
                    else
                        break;
                    strPtr += 2;
                }
                pname->font_family_name[jj] = 0;
            }
            else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_UniqueName)
            {
                /* if font unique name length is larger than pre-allocated
                 * buffer length, set "too_long" flag
                 */
                if (strLen > MAX_FONT_NAME_LEN - 1)
                    pname->unique_name_too_long = true;

                /* strings are in Unicode - 2 bytes per char */

                for (jj = 0, strPtr++; jj < strLen; jj++)
                {
                    if (jj < (MAX_FONT_NAME_LEN - 1))
                        pname->unique_name[jj] = *strPtr;
                    else
                        break;
                    strPtr += 2;
                }
                pname->unique_name[jj] = 0;
            }
            else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_FullName)
            {
                /* if font name length is larger than pre-allocated
                 * buffer length, set "too_long" flag
                 */
                if (strLen > MAX_FONT_NAME_LEN - 1)
                    pname->font_name_too_long = true;

                /* strings are in Unicode - 2 bytes per char */

                for (jj = 0, strPtr++; jj < strLen; jj++)
                {
                    if (jj < (MAX_FONT_NAME_LEN - 1))
                        pname->font_name[jj] = *strPtr;
                    else
                        break;
                    strPtr += 2;
                }
                pname->font_name[jj] = 0;
            }
            else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_Copyright)
            {
                /* if copyright length is larger than pre-allocated
                 * buffer length, set "too_long" flag
                 */
                if (strLen > MAX_COPYRIGHT_LEN - 1)
                    pname->copyright_too_long = true;

                /* strings are in Unicode - 2 bytes per char */

                for (jj = 0, strPtr++; jj < strLen; jj++)
                {
                    if (jj < (MAX_COPYRIGHT_LEN - 1))
                        pname->copyright[jj] = *strPtr;
                    else
                        break;
                    strPtr += 2;
                }
                pname->copyright[jj] = 0;
            }
        }
        /* exit loop once we got names */
        if (pname->font_name[0] && pname->font_family_name[0] &&
                pname->copyright[0])
            break;
    }

    /* if we did not got all names */
    /* extract the font name, family name and copyright data from the "name" table - MAC format */
    if ( ( pname->font_name[0] == 0 ) ||
         ( pname->font_family_name[0] == 0 ) ||
         ( pname->copyright[0] == 0 ) )
    {
        ptr = p + NAME_NAMERECS;                  /* The NameRecords */
        for (ii = 0; ii < ulnrec; ii++, ptr += NAME_SIZE_NAMEREC)
        {
            if ((GET_xWORD((ptr + NAME_TAB_PLATID))) == MAC_ENC  &&
                (GET_xWORD((ptr + NAME_TAB_LANGID))) == MAC_US_ENGL )
            {
                FS_BYTE *strPtr = p + ostr + GET_xWORD((ptr + NAME_TAB_STROFF));
                FS_SHORT strLen = GET_xWORD((ptr + NAME_TAB_STRLEN));
                if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_Family)
                {
                    /* if font family name length is larger than pre-allocated
                     * buffer length, set "too_long" flag
                     */
                    if (strLen > MAX_FONT_FAMILY_NAME_LEN - 1)
                        pname->font_family_name_too_long = true;

                    for (jj = 0; jj < strLen; jj++)
                    {
                        if (jj < (MAX_FONT_FAMILY_NAME_LEN - 1))
                            pname->font_family_name[jj] = *strPtr;
                        else
                            break;
                        strPtr++;
                    }
                    pname->font_family_name[jj] = 0;
                }
                else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_UniqueName)
                {
                    /* if font unique name length is larger than pre-allocated
                     * buffer length, set "too_long" flag
                     */
                    if (strLen > MAX_FONT_NAME_LEN - 1)
                        pname->unique_name_too_long = true;

                    for (jj = 0; jj < strLen; jj++)
                    {
                        if (jj < (MAX_FONT_NAME_LEN - 1))
                            pname->unique_name[jj] = *strPtr;
                        else
                            break;
                        strPtr++;
                    }
                    pname->unique_name[jj] = 0;
                }
                else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_FullName)
                {
                    /* if font name length is larger than pre-allocated
                     * buffer length, set "too_long" flag
                     */
                    if (strLen > MAX_FONT_NAME_LEN - 1)
                        pname->font_name_too_long = true;

                    for (jj = 0; jj < strLen; jj++)
                    {
                        if (jj < (MAX_FONT_NAME_LEN - 1))
                            pname->font_name[jj] = *strPtr;
                        else
                            break;
                        strPtr++;
                    }
                    pname->font_name[jj] = 0;
                }
                else if ((GET_xWORD((ptr + NAME_TAB_NAMEID))) == name_Copyright)
                {
                    /* if copyright length is larger than pre-allocated
                     * buffer length, set "too_long" flag
                     */
                    if (strLen > MAX_COPYRIGHT_LEN - 1)
                        pname->copyright_too_long = true;

                    for (jj = 0; jj < strLen; jj++)
                    {
                        if (jj < (MAX_COPYRIGHT_LEN - 1))
                            pname->copyright[jj] = *strPtr;
                        else
                            break;
                        strPtr++;
                    }
                    pname->copyright[jj] = 0;
                }
            }
            /* exit loop once we got all names */
            if (pname->font_name[0] && pname->font_family_name[0] &&
                pname->copyright[0])
                break;
        }
    }

    /* check whether names were found. If not, provide names */
    if (pname->font_name[0] == 0)
        SYS_STRNCPY(pname->font_name, (FS_CONST char *)"NOT FOUND", 9);

    if (pname->font_family_name[0] == 0)
        SYS_STRNCPY(pname->font_family_name, (FS_CONST char *)"NOT FOUND", 9);

    if (pname->copyright[0] == 0)
        SYS_STRNCPY(pname->copyright, (FS_CONST char *)"NOT FOUND", 9);

    STATE.error = SUCCESS;
    return pname;
}

TTF_NAME *get_name(_DS_ TTF *ttf)
{
    FS_BYTE *p;
    TTF_NAME *pname;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_NAME";
#endif

    /* read the entire "name" table into temporary buffer 'p' */
    p = get_ttf_table(_PS_ ttf, TAG_name);
    if ((p == 0) || STATE.error)
        return (TTF_NAME *)0;

    pname = get_abbreviated_name(_PS_ p);

    /* free the "name" table buffer */
    FSS_free(_PS_ p);

    return pname;
}

/****************************************************************/
FS_VOID swap_ttc_header(TTC_HEADER *p)
{
    FS_ULONG i;

    p->tag = SWAPL(p->tag);
    p->version = SWAPL(p->version);
    p->numFonts = SWAPL(p->numFonts);
    for (i = 0; i < p->numFonts; i++)
        p->offsets[i] = SWAPL(p->offsets[i]);
}

/****************************************************************/
FS_VOID swap_ttf_header(TTF_HEADER *p)
{
    FS_LONG i, num, version;
    TTF_DIR *q;

    version = p->version;
    if (version == 0x00010000 || version == 0x00020000 || version == 0x74727565)
        num = p->numTables;   /* already in OS byte order, so take numTables directly */
    else
        num = SWAPW(p->numTables); /* not in OS byte order, so need to swap numTables */

    if (num > 50) return; /* satisfy Klocwork */

    p->version = SWAPL(p->version);
    p->numTables = SWAPW(p->numTables);
    p->searchRange = SWAPW(p->searchRange);
    p->entrySelector = SWAPW(p->entrySelector);
    p->rangeShift = SWAPW(p->rangeShift);
    for (i = 0; i < num; i++)
    {
        q = &p->tables[i];
        q->tag = SWAPL(q->tag);
        q->checkSum = SWAPL(q->checkSum);
        q->offset = SWAPL(q->offset);
        q->length = SWAPL(q->length);
    }
}

/****************************************************************/
FS_VOID swap_head(TTF_HEAD *p)
{
    p->version = SWAPL(p->version);
    p->fontRevision = SWAPL(p->fontRevision);
    p->checkSumAdjustment = SWAPL(p->checkSumAdjustment);
    p->magicNumber = SWAPL(p->magicNumber);
    p->flags = SWAPW(p->flags);
    p->unitsPerEm = SWAPW(p->unitsPerEm);
    p->created[0] = SWAPL(p->created[0]);
    p->created[1] = SWAPL(p->created[1]);
    p->modified[0] = SWAPL(p->modified[0]);
    p->modified[1] = SWAPL(p->modified[1]);
    p->xMin = SWAPW(p->xMin);
    p->yMin = SWAPW(p->yMin);
    p->xMax = SWAPW(p->xMax);
    p->yMax = SWAPW(p->yMax);
    p->macStyle = SWAPW(p->macStyle);
    p->lowestRecPPEM = SWAPW(p->lowestRecPPEM);
    p->fontDirectionHint = SWAPW(p->fontDirectionHint);
    p->indexToLocFormat = SWAPW(p->indexToLocFormat);
    p->glyphDataFormat = SWAPW(p->glyphDataFormat);
}

/****************************************************************/
FS_VOID swap_hhea(TTF_HHEA *p)
{
    p->version = SWAPL(p->version);
    p->yAscender = SWAPW(p->yAscender);
    p->yDescender = SWAPW(p->yDescender);
    p->yLineGap = SWAPW(p->yLineGap);
    p->advanceWidthMax = SWAPW(p->advanceWidthMax);
    p->minLeftSideBearing = SWAPW(p->minLeftSideBearing);
    p->minRightSideBearing = SWAPW(p->minRightSideBearing);
    p->xMaxExtent = SWAPW(p->xMaxExtent);
    p->horizontalCaretSlopeNumerator = SWAPW(p->horizontalCaretSlopeNumerator);
    p->horizontalCaretSlopeDenominator = SWAPW(p->horizontalCaretSlopeDenominator);
    p->caretOffset = SWAPW(p->caretOffset);
    p->metricDataFormat = SWAPW(p->metricDataFormat);
    p->numberOf_LongHorMetrics = SWAPW(p->numberOf_LongHorMetrics);
}

/****************************************************************/
FS_VOID swap_vhea(TTF_VHEA *p)
{
    p->version = SWAPL(p->version);
    p->yAscender = SWAPW(p->yAscender);
    p->yDescender = SWAPW(p->yDescender);
    p->yLineGap = SWAPW(p->yLineGap);
    p->advanceHeightMax = SWAPW(p->advanceHeightMax);
    p->minTopSideBearing = SWAPW(p->minTopSideBearing);
    p->minBottomSideBearing = SWAPW(p->minBottomSideBearing);
    p->yMaxExtent = SWAPW(p->yMaxExtent);
    p->horizontalCaretSlopeNumerator = SWAPW(p->horizontalCaretSlopeNumerator);
    p->horizontalCaretSlopeDenominator = SWAPW(p->horizontalCaretSlopeDenominator);
    p->caretOffset = SWAPW(p->caretOffset);
    p->metricDataFormat = SWAPW(p->metricDataFormat);
    p->numberOf_LongVerMetrics = SWAPW(p->numberOf_LongVerMetrics);
}

/****************************************************************/
FS_VOID swap_maxp(TTF_MAXP *p)
{
    p->version = SWAPL(p->version);
    p->numGlyphs = SWAPW(p->numGlyphs);
    p->maxPoints = SWAPW(p->maxPoints);
    p->maxContours = SWAPW(p->maxContours);
    p->maxCompositePoints = SWAPW(p->maxCompositePoints);
    p->maxCompositeContours = SWAPW(p->maxCompositeContours);
    p->maxElements = SWAPW(p->maxElements);
    p->maxTwilightPoints = SWAPW(p->maxTwilightPoints);
    p->maxStorage = SWAPW(p->maxStorage);
    p->maxFunctionDefs = SWAPW(p->maxFunctionDefs);
    p->maxInstructionDefs = SWAPW(p->maxInstructionDefs);
    p->maxStackElements = SWAPW(p->maxStackElements);
    p->maxSizeOfInstructions = SWAPW(p->maxSizeOfInstructions);
    p->maxComponentElements = SWAPW(p->maxComponentElements);
    p->maxComponentDepth = SWAPW(p->maxComponentDepth);
}
/****************************************************************/
